/*
 * Worldcup is Copyright (c) 2001-2006 by Thomas Pundt
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without a written agreement is hereby granted,
 * provided that the above copyright notice and this paragraph and the
 * following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 * THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <PalmOS.h>

#include "worldcup.h"
#include "worldcuprcp.h"

struct matchfield match[8];

struct teamstate Teamstate[4], Equal[4];

static UInt8 rounds[5] = { 0, 8, 12, 14, 16 };
// static UInt8 lots[32] = { 0, };

MyAppPreferencesType appPrefs;
Int8 selectedMatch = -1;
UInt8 selectedTeam = 0;
Int32 defaultDateFormat;

static TablePtr leagueTable;

/**
 * Credits and other Help forms (via (i) in About Form)
 */
void InitFrmHelp(UInt16 msgID, UInt16 titleID)
{
  FormPtr frm = FrmGetActiveForm();
  FieldPtr fld = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, IdFrmHelp));
  static char* msg;
  static char* title;
  msg = GetString(msgID);
  title = GetString(titleID);
  FrmSetTitle(frm,title);
  FldSetTextPtr(fld, msg);
  FldRecalculateField(fld, true);
  FldRecalculateScrollbar(fld, Id2Ptr(IdFrmHelp+1));
}

Boolean FrmHelpHandleEvent (EventPtr e)
{
  Boolean handled = false;
  
  switch (e->eType) {
  case sclRepeatEvent:
    FldDoScrolling(Id2Ptr(IdFrmHelp), Id2Ptr(IdFrmHelp+1));
    break;
  
  default:
    ;
  }
  return handled;
}

static void SetDateString(Int16 id, UInt32 datum)
{
  DateTimeType dateP;
  ControlType *ctlP = Id2Ptr(id);
  char *label = (char*)CtlGetLabel(ctlP);
  CtlEraseControl(ctlP);

  TimSecondsToDateTime(datum+appPrefs.gmtOff, &dateP);
  DateToAscii(dateP.month,dateP.day,dateP.year,
	      PrefGetPreference(prefDateFormat), label);

  // don't draw 0:00 time
  if (dateP.hour+dateP.minute>0) {
    StrCat(label," ");
    TimeToAscii(dateP.hour,dateP.minute,tfColon24h,
		label+StrLen(label));
  }
  CtlDrawControl(ctlP);
}

static void SetRoundString(Int16 id, int num)
{
  char *p, *label;
  ControlType *ctlP = Id2Ptr(id);
  label = (char*)CtlGetLabel(ctlP);
  p = GetString(num);
  CtlEraseControl(ctlP);
  StrCopy(label,p);
  CtlDrawControl(ctlP);
}

static void SetTimezoneString(Int16 id, int num, const Char* format)
{
  char *label;
  ControlType *ctlP = Id2Ptr(id);
  label = (char*)CtlGetLabel(ctlP);
  StrPrintF(label,format,num);
  CtlEraseControl(ctlP);
  CtlDrawControl(ctlP);
}

/*
 * This comparison function takes into account items 6.a)-6.c) from
 * Article 28) of FIFA Worldcup 2002 Reglement or items 6.d)-6.g)
 * (points, goals and goals difference equality) if executed from a
 * second call to ComputeTable (with a smaller table).
 */
static char cmpTeamsByPtsGls(struct teamstate *a, struct teamstate *b)
{
  if (a->Points > b->Points) return 1;
  if (a->Points < b->Points) return -1;
  if (a->Goals + b->OppGoals > b->Goals + a->OppGoals) return 1;
  if (a->Goals + b->OppGoals < b->Goals + a->OppGoals) return -1;
  if (a->Goals > b->Goals) return 1;
  if (a->Goals < b->Goals) return -1;
  return 0;
}

static void AddTeamstate(struct teamstate *table, Int8 n, Int8 n1, Int8 n2)
{
  Int8 pts = 3;
  if (n>0) n--;

  table[n].Goals  += n1;
  table[n].OppGoals += n2;
  table[n].Matches++;
  if (n1>n2) {
    table[n].Points+=pts;
    table[n].Won++;
  } else if (n1==n2) {
    table[n].Points++;
    table[n].Draw++;
  } else {
    table[n].Lost++;
  }
}

/*
 * compute table;
 * table:     standings table
 * who:       bit field of teams to compute table for
 * recursive: flag if we are called recursive; happens while
 *            computing a sub table when points, goals and goal
 *            difference for some teams are equal
 * returns:   "final" flag.
 */
Int8 ComputeTable(struct teamstate *table, struct matchfield *match,
		  Int8 who, Int8 recursive)
{
  Int16 i,j;
  Int8 final=1, positions[4] = { 0, }, onpos[4] = { 0, };

  // init variables;
  for (i=0; i<4; i++) {
    table[i].Goals = table[i].OppGoals = table[i].Points = table[i].Matches
      = table[i].Won = table[i].Draw = table[i].Lost = 0;
    table[i].Pos = 1;
  }

  // take every match of every team within "who"
  for (i=0; i<6; i++) {
    if (((1<<(match[i].team1-1)) & who) && ((1<<(match[i].team2-1)) & who) ) {
      if ( (match[i].r1 != -1) && (match[i].r2 != -1) ) {
	AddTeamstate(table,match[i].team1,match[i].r1,match[i].r2);
	AddTeamstate(table,match[i].team2,match[i].r2,match[i].r1);
      } else {
	// there is at least one match missing for a final table.
	final = 0;
      }
    }
  }

  // actually compute table for every team within "who"
  for (i=0; i<3; i++)
    for (j=i+1; j<4; j++)
      if ((1<<i & who) && ((1<<j & who))) {
	char sort = cmpTeamsByPtsGls(&table[i],&table[j]);
	if (sort<0)
	  table[i].Pos++;
	else if (sort>0)
	  table[j].Pos++;
      }

  // if we have all results, check if some teams are equally placed.
  // If so, compute a table with all matches of these teams. Required
  // by Article 28) of FIFA WC 2002 Reglement.
  if (final) {

    // first look for equally placed teams
    for (i=0; i<4; i++) {
      if ( (1<<i & who) ) {
	positions[table[i].Pos-1]++; // count teams on place
	onpos[table[i].Pos-1] |= 1<<i; // remember team
      }
    }

    // if we have some and haven't compared the matches between
    // them yet, do this now.
    if (!recursive) {
      for (i=0; i<4; i++) {
	if (positions[i]>1) { // at least two teams with equal place
	  final += ComputeTable(Equal,match,onpos[i],1);
	  for (j=0; j<4; j++) { // evaluate newly computed table
	    table[j].Pos += Equal[j].Pos-1;
	  }
	  // Losentscheid hier verankern...
	}
      }
    } else {
      /* we are in the process to compare the matches between
       * equally placed teams; if the table here also reveals
       * equality between teams, simply return this fact;
       * situation for drawing lots */
      for (i=0; i<4; i++) {
	if (positions[i]>1) {
	  return 1;
	}
      }
      return 0;
    }
  }
  return final;
}

/**
 * Form Compute Table Functions
 */

static void DrawTableItem(void* tableP, Int16 row, Int16 column,
			  RectangleType *bounds)
{
  Int16 n;
  UInt16 x;
  Char buf[25];
  struct team *team;

  n = TblGetItemInt(tableP, row, column);
  x = bounds->topLeft.x;

  StrIToA(buf,n);

  switch (column) {
  case 0:
    StrCat(buf,".");
    break;
  case 1:
    team = GetTeamString((appPrefs.group-1)*4+n);
    MemMove(buf,team->name,StrLen(team->name)+1);
    break;
  case 3:
    StrCat(buf," :");
    x += bounds->extent.x - FntCharsWidth(buf,StrLen(buf)) - 1;
    break;
  }

  FntSetFont(0);
  WinEraseRectangle(bounds, 0);
  WinDrawChars(buf, StrLen(buf), x, bounds->topLeft.y);
}

/*
 * fills in the values for the table and redraws it.
 */
static void InitTable(void)
{
  Int16 Pos, n, row=0;

  for (Pos=1; Pos<=4; Pos++) {
    for (n=0; n<4; n++) {
      if (Teamstate[n].Pos == Pos) {
        TblSetItemInt(leagueTable, row, 0, Pos);
        TblSetItemInt(leagueTable, row, 1, n+1);
        TblSetItemInt(leagueTable, row, 2, Teamstate[n].Matches);
        TblSetItemInt(leagueTable, row, 3, Teamstate[n].Goals);
        TblSetItemInt(leagueTable, row, 4, Teamstate[n].OppGoals);
        TblSetItemInt(leagueTable, row, 5, Teamstate[n].Points);
        row++;
      }
    }
  }
}

/*
 * get the contents of the editable fields in the form and save them
 */
static void SaveMatchdayResults(void)
{
  Int16 i;
  char *r1, *r2;
  Int8 buf[2], rows;

  for (i=0; i<8; i++) {
    r1 = FldGetTextPtr(match[i].r1fld);
    r2 = FldGetTextPtr(match[i].r2fld);

    buf[0] = buf[1] = 255;

    if (r1 && StrLen(r1)
	&& r2 && StrLen(r2)) {
      buf[0] = StrAToI(r1);
      buf[1] = StrAToI(r2);
    }
    match[i].r1 = buf[0];
    match[i].r2 = buf[1];
  }

  if (appPrefs.group > 0) rows = 6;
  else rows = rounds[appPrefs.round+1]-rounds[appPrefs.round];
  WriteFixture(match, rows, appPrefs.group);
}

void ReadSetFixture(struct matchfield* match, UInt8 group, Boolean setallfields)
{
  struct match Match;
  char *p;
  UInt8 i = 0, j = 0, recNo;
  for (p=ReadFixture(0, &Match),recNo=0; p; p=ReadFixture(p, &Match),recNo++) {
    if (Match.group != group)
      continue;
    if (group == 0) {
      j++;
      if (j<=rounds[appPrefs.round] || j>rounds[appPrefs.round+1])
	continue;
    }
    match[i].date = Match.date;
    match[i].venue = Match.venue;
    match[i].r1 = Match.r1;
    match[i].r2 = Match.r2;
    match[i].team1 = Match.team1;
    match[i].team2 = Match.team2;
    match[i].recNo = recNo;
    if (setallfields) {
      if (group>0) {
	SetHandleString(match[i].team1fld, (group-1)*4 + Match.team1-1);
	SetHandleString(match[i].team2fld, (group-1)*4 + Match.team2-1);
	match[i].team1Final = Match.team1+(group-1)*4;
	match[i].team2Final = Match.team2+(group-1)*4;
      } else {
	match[i].team1Final = ComputeSetHandleString(match[i].team1fld, Match.team1);
	match[i].team2Final = ComputeSetHandleString(match[i].team2fld, Match.team2);
	SetHandleInt(match[i].num,recNo+1);
      }
      SetHandleInt(match[i].r1fld,Match.r1);
      SetHandleInt(match[i].r2fld,Match.r2);
    }
    i++;
  }
}

Int8 InitFormMain(void)
{
  Int16 i, j, rows;
  Int8 final = 0;
  FormType* frm = FrmGetActiveForm();

  selectedMatch = -1;
  selectedTeam = 0;

  if (appPrefs.group > 0) 
    rows = 6;
  else 
    rows = rounds[appPrefs.round+1]-rounds[appPrefs.round];

  /*
   * initialize table variables to default values;
   */
  leagueTable = Id2Ptr(IdFrmMain+100);

  for (i=0; i<8; i++) {
    match[i].team1fld = Id2Ptr(IdFrmMain+20+i);
    match[i].team2fld = Id2Ptr(IdFrmMain+30+i);
    match[i].r1fld = Id2Ptr(IdFrmMain+40+i);
    match[i].r2fld = Id2Ptr(IdFrmMain+50+i);
    match[i].num = Id2Ptr(IdFrmMain+60+i);
    FldSetUsable(match[i].team1fld, i<rows?true:false);
    FldSetUsable(match[i].team2fld, i<rows?true:false);
    FldSetUsable(match[i].r1fld, i<rows?true:false);
    FldSetUsable(match[i].r2fld, i<rows?true:false);
    FldSetUsable(match[i].num, i<rows?true:false);
    CtlSetUsable(Id2Ptr(IdFrmMain+10+i), i<rows?true:false);
  }

  ReadSetFixture(match,appPrefs.group,true);
  if (appPrefs.group>0) {
    FrmHideObject(frm, FrmGetObjectIndex(frm,IdFrmMain+3));
    FrmHideObject(frm, FrmGetObjectIndex(frm,IdFrmMain+4));
    for (i=0; i<4; i++) {
      TblSetRowSelectable(leagueTable, i, false);
      TblSetRowUsable(leagueTable, i, appPrefs.group>0?true:false);
      for (j=0; j<6; j++) {
	TblSetItemStyle(leagueTable, i, j, customTableItem);
	TblSetColumnUsable(leagueTable, j, true);
	TblSetCustomDrawProcedure(leagueTable, j, DrawTableItem);
      }
    }
    final = ComputeTable(Teamstate,match,15,0); // 15 => all bits set, all teams
    InitTable();
  } else {
    FrmHideObject(frm, FrmGetObjectIndex(frm,IdFrmMain+2));
    FrmHideObject(frm, FrmGetObjectIndex(frm,IdFrmMain+7));
    FrmHideObject(frm, FrmGetObjectIndex(frm,IdFrmMain+8));
    FrmSetObjectPosition(frm, FrmGetObjectIndex(frm,IdFrmMain+100),0,159);
  }
  FrmCopyLabel(frm, IdFrmMain,
	       LstGetSelectionText(Id2Ptr(IdFrmMain+1),appPrefs.group));
  FrmDrawForm(frm);

  DrawButton(IdFrmMain+91,IdBitmap+11);
  DrawButton(IdFrmMain+92,IdBitmap+12);
  DrawButton(IdFrmMain+93,IdBitmap+14);

  SetRoundString(IdFrmMain+5,IdRoundString+(appPrefs.group>0 ? 0 : appPrefs.round+1));
  return final;
}

static RectangleType eRect;
static PointType ePt;

static void dragTeam(Coord x, Coord y)
{
  WinInvertRectangle(&eRect, 2);
  RctOffsetRectangle(&eRect, x-ePt.x, y-ePt.y);
  WinInvertRectangle(&eRect, 2);
  ePt.x = x;
  ePt.y = y;
}

static Int16 teamUnder(Coord x, Coord y)
{
  if (y<115 || x>98 || x<16)
    return -1;
  y = (y-115)/11;
  TblGetItemBounds(leagueTable,y,1,&eRect);
  return y;
}

Boolean SpieltagFormHandleEvent(EventPtr e)
{
  Boolean handled = false;
  Int16 item;
  ListPtr pLst;
  static Int16 lastTeam;
  static Int8 final;

  switch (e->eType) {

  case frmOpenEvent:
    final = InitFormMain();
    handled = true;
    break;

  case menuEvent:

    switch(e->data.menu.itemID) {

    case IdMenuItem:
      FrmPopupForm(IdFrmAbout);
      break;

    case IdMenuItem+1:
      MyDoDialog(IdFrmHelp, FrmHelpHandleEvent, InitFrmHelp, IdFrmMain, IdStringTitle+1);
      break;

    case IdMenuItem+2:
      FrmPopupForm(IdFrmTimezone);
      break;

    case IdMenuItem+4:
      SaveMatchdayResults();
      FrmGotoForm(IdFrmMatches);
      break;

    case IdMenuItem+5:
      BeamDatabase();
      break;

    case IdMenuItem+6:
      SaveMatchdayResults();
      FrmGotoForm(IdFrmFixture);
      break;

    }
    handled = true;
    break;

  case penDownEvent:
    /* are we over a field with a team? */
    if ( final > 1 && (lastTeam = teamUnder(e->screenX, e->screenY)) >= 0 ) {
      WinInvertRectangle(&eRect, 2);
      ePt.x = e->screenX;
      ePt.y = e->screenY;
      handled = true;
    }
    break;

  case penMoveEvent:
    /* move rectangle */
    if ( final > 1 && lastTeam>=0 ) {
      dragTeam(e->screenX, e->screenY);
      handled = true;
    }
    break;

  case penUpEvent:
    /* swap teams */
    if ( final > 1 && lastTeam>= 0 ) {
      Int16 newTeam;
      WinInvertRectangle(&eRect, 2);
      if ( (newTeam = teamUnder(e->screenX, e->screenY)) >= 0 ) {
	// exchange(lastTeam,newTeam);
        // hasChanged = true;
	handled = true;
      }
    }
    break;

  case ctlSelectEvent:
    switch (e->data.ctlEnter.controlID) {

    popup:
    case IdFrmMain:
      pLst = Id2Ptr(IdFrmMain+ (appPrefs.options & OPT_AlternativeGroupLabels?9:1));
      LstSetSelection(pLst, appPrefs.group);
      LstMakeItemVisible(pLst, appPrefs.group);
      LstPopupList(pLst);
      if ( (item = LstGetSelection(pLst)) != appPrefs.group && item!=-1) {
	SaveMatchdayResults();
	appPrefs.group = item;
	FrmGotoForm(IdFrmMain);
      }
      handled = true;
      break;

    case IdFrmMain+2: /* Down-Arrow => Compute Table */
      SaveMatchdayResults();
      final = ComputeTable(Teamstate,match,15,0); // 15 => all bits set, all teams
      InitTable();
      TblDrawTable(leagueTable);
      handled = true;
      break;

    case IdFrmMain+3: /* Left-Arrow  */
      if (appPrefs.round>0) {
	SaveMatchdayResults();
	appPrefs.round--;
	FrmGotoForm(IdFrmMain);
      }
      handled = true;
      break;

    case IdFrmMain+4: /* Right-Arrow */
      if (appPrefs.round<3) {
	SaveMatchdayResults();
	appPrefs.round++;
	FrmGotoForm(IdFrmMain);
      }
      handled = true;
      break;

    case IdFrmMain+92: // "List form" button
      SaveMatchdayResults();
      FrmGotoForm(IdFrmMatches);
      break;

    case IdFrmMain+93: // "Overview form" button
      SaveMatchdayResults();
      FrmGotoForm(IdFrmFixture);
      break;

    default:
      /* (i) button => show popup with place and date of match */
      if (e->data.ctlEnter.controlID >= IdFrmMain+10 &&
	  e->data.ctlEnter.controlID <= IdFrmMain+17) {
	int index = e->data.ctlEnter.controlID - IdFrmMain - 10;
	selectedMatch = index;
	FrmPopupForm(IdFrmMainPopup);
	handled = true;
      }
    }
    break;

  case fldEnterEvent:
    item = e->data.fldEnter.fieldID-IdFrmMain-20;
    if (item>=0 && item<8) {
      handled = true;
      selectedTeam = match[item].team1Final;
    } else if (item>=10 && item<18) {
      handled = true;
      item -= 10;
      selectedTeam = match[item].team2Final;
    }
    if (handled && selectedTeam>0 && selectedTeam<=32)
      FrmPopupForm(IdFrmTeams);
    else if (e->data.fldEnter.fieldID >= IdFrmMain+40 &&
	     e->data.fldEnter.fieldID <= IdFrmMain+57) {
      // Compute table whenever you enter another field.
      FormPtr frm = FrmGetActiveForm();
      SaveMatchdayResults();
      final = ComputeTable(Teamstate,match,15,0); // 15 => all bits set, all teams
      InitTable();
      TblDrawTable(leagueTable);
      FrmSetFocus(frm, FrmGetObjectIndex(frm,e->data.fldEnter.fieldID));
      handled = true;
    }
    break;

  case keyDownEvent:
    if (e->data.keyDown.chr == vchrPageUp) {
      SaveMatchdayResults();
      if (appPrefs.group == 0 && appPrefs.round > 0) {
        appPrefs.round--;
      } else {
        appPrefs.group--;
        if (appPrefs.group == 0) appPrefs.round = 3;
      }
      if (appPrefs.group < 0) {
        appPrefs.group = 8;
      }
      FrmGotoForm(IdFrmMain);
      handled = true;
    }
              
    if (e->data.keyDown.chr == vchrPageDown) {
      SaveMatchdayResults();
      if (appPrefs.group == 0 && appPrefs.round < 3) {
        appPrefs.round++;
      } else {
        appPrefs.group++;
      }
      if (appPrefs.group > 8) {
        appPrefs.group = 0;
        appPrefs.round = 0;
      }
      FrmGotoForm(IdFrmMain);
      handled = true;
      break;
    }

    if (e->data.keyDown.chr == vchrRockerLeft ||
        e->data.keyDown.keyCode == fiveWayLeft) {
      FrmGotoForm(IdFrmFixture);
      handled = true;
    }

    if (e->data.keyDown.chr == vchrRockerRight ||
        e->data.keyDown.keyCode == fiveWayRight) {
      FrmGotoForm(IdFrmMatches);
      handled = true;
    }

    if (e->data.keyDown.chr == vchrRockerCenter ||
        (e->data.keyDown.keyCode == fiveWaySelect &&
         (e->data.keyDown.modifiers & autoRepeatKeyMask) == 0) ) {
      goto popup;
    }

    break;

  default:
    break;
  }

  return handled;
}

Boolean MainPopupFormHandleEvent(EventPtr e)
{
  Boolean handled = false;
  struct venue* venue;

  switch (e->eType) {
  case frmOpenEvent:
    // display the team names
    if (appPrefs.group>0) { // first round
      SetHandleString(Id2Ptr(IdFrmMainPopup), 
                      (appPrefs.group-1)*4 + match[selectedMatch].team1 - 1);
      SetHandleString(Id2Ptr(IdFrmMainPopup + 2), 
                      (appPrefs.group-1)*4 + match[selectedMatch].team2 -1);
    } else { // group == 0 --> second round
      ComputeSetHandleString(Id2Ptr(IdFrmMainPopup), 
                             match[selectedMatch].team1);
      ComputeSetHandleString(Id2Ptr(IdFrmMainPopup + 2), 
                             match[selectedMatch].team2);
    }

    FrmDrawForm(FrmGetActiveForm());
    venue = GetLocation(match[selectedMatch].venue-1); // display stadium
    ButtonSetLabel(IdFrmMainPopup+5,venue->name);
    SetDateString(IdFrmMainPopup + 6, match[selectedMatch].date); // display time of match

    handled = true;
    break;

  case ctlSelectEvent:
    switch (e->data.ctlEnter.controlID) {

    case IdFrmMainPopup+5: // Stadium Button
      if (selectedMatch != -1)
	FrmPopupForm(IdFrmStadium);
      handled = true;
      break;

    case IdFrmMainPopup+6: // Date Button
      if ( selectedMatch != -1 &&
	   0 == DisplayConfirmationMessage(IdErrorMsg+4)) {
	if (OpenDatebook()) {
	  MakeDatebookEntry(match[selectedMatch].date + appPrefs.gmtOff,
			    (match[selectedMatch].team1Final<<8)+
			    match[selectedMatch].team2Final);
	  DmCloseDatabase(datebook);
	}
      }
      handled = true;
      break;
    }
    break;

  case penDownEvent:
    if (e->screenY > 27) { // User taps outside the popup --> dismiss it
      // 27 = hard coded form height. Should be obtained dynamically.
      FrmReturnToForm(0);
      handled = true;
    }
    break;

  default:
    break;
  }

  return handled;
}

/**
 * About Form
 */
Boolean AboutFormHandleEvent(EventPtr e)
{
  Boolean handled;

  handled = false;
  switch (e->eType) {

  case frmOpenEvent:
    FrmDrawForm(FrmGetActiveForm());
    handled = true;
    break;

  case ctlSelectEvent:
    if (e->data.ctlSelect.controlID == IdFrmAbout) {
      FrmReturnToForm(0);
      handled = true;
    }
    break;

  case penDownEvent:
    // this is the position of the (i)
    if (e->screenX >= 144 && e->screenX <= 155 &&
	e->screenY >= 0 && e->screenY <= 11) {
      // this way we don't have to use the default bold font.
      MyDoDialog(IdFrmHelp, FrmHelpHandleEvent, InitFrmHelp, IdFrmAbout, IdStringTitle);
      handled =true;
    }

  default:
    break;
  }

  return handled;
}

/**
 * Timezone and Options Form
 */
Boolean TimezoneFormHandleEvent(EventPtr e)
{
  static UInt8 element;
  static Int32 newGmtOff, minutes;
  static Boolean timezoneChanged;
  Boolean handled;
  FormPtr frm;

  handled = false;
  switch (e->eType) {

  case frmOpenEvent:
    element = 0;
    timezoneChanged = false;
    // just to be sure, that the date really is correctly "aligned"
    newGmtOff = (appPrefs.gmtOff/1800L)*1800L;
    minutes = (newGmtOff%3600L)/60L;
    frm = FrmGetActiveForm();
    FrmSetControlGroupSelection(frm, 1, IdFrmTimezone+2);
    FrmSetControlValue(frm, FrmGetObjectIndex(frm, IdFrmTimezone+6),
		       (appPrefs.options & OPT_UseFrmMatches));
    FrmSetControlValue(frm, FrmGetObjectIndex(frm, IdFrmTimezone+7),
		       (appPrefs.options & OPT_AlternativeGroupLabels));

    FrmDrawForm(frm);
    SetTimezoneString(IdFrmTimezone+2,newGmtOff/3600L,"%+2d");
    SetTimezoneString(IdFrmTimezone+3,minutes,"%02d");

    if ( FeatureSet35 ) {
      WinPushDrawState();
      WinSetForeColor(UIColorGetTableEntryIndex(UIDialogFrame));
    }
    WinDrawLine(0,46,159,46);
    WinDrawLine(0,47,159,47);
    if ( FeatureSet35 ) {
      WinPopDrawState();
    }

    handled = true;
    break;

  case ctlSelectEvent:
    switch (e->data.ctlSelect.controlID) {
    case IdFrmTimezone:
      appPrefs.gmtOff = newGmtOff;
    case IdFrmTimezone+1:
      FrmReturnToForm(0);
      if (timezoneChanged)
        FrmGotoForm(FrmGetFormId(FrmGetActiveForm()));
      handled = true;
      break;

    case IdFrmTimezone+2:
      element = 0;
      handled = true;
      break;
    case IdFrmTimezone+3:
      element = 1;
      handled = true;
      break;

    case IdFrmTimezone+4:
      handled = true;
      timezoneChanged = true;
      /* according to a time zone map there really _is_ a time zone
       * that's off by +1400...
       */
      if (element == 0) {
	if (newGmtOff<=13L*3600L) {
	  newGmtOff += 3600L;
	  SetTimezoneString(IdFrmTimezone+2,newGmtOff/3600L,"%+2d");
	}
	break;
      }
    case IdFrmTimezone+5:
      handled = true;
      timezoneChanged = true;
      if (element == 1) { // doesn't matter if down or up...
	newGmtOff -= 60L*minutes;
	minutes = (30-minutes);
	newGmtOff += 60L*minutes;
	SetTimezoneString(IdFrmTimezone+3,minutes,"%02d");
	break;
      }
      if (element == 0 && newGmtOff>=-11L*3600L) {
	newGmtOff -= 3600L;
	SetTimezoneString(IdFrmTimezone+2,newGmtOff/3600L,"%+2d");
	break;
      }
      break;

    // the new options checkboxes
    case IdFrmTimezone+6:
      appPrefs.options ^= OPT_UseFrmMatches;
      handled = true;
      break;

    case IdFrmTimezone+7:
      appPrefs.options ^= OPT_AlternativeGroupLabels;
      handled = true;
      break;

    }
    break;

  case penDownEvent:
    // this is the position of the (i)
    if (e->screenX >= 144 && e->screenX <= 155 &&
	e->screenY >= 0 && e->screenY <= 11) {
      // this way we don't have to use the default bold font.
      MyDoDialog(IdFrmHelp, FrmHelpHandleEvent, InitFrmHelp, IdFrmTimezone, IdStringTitle+1);
      handled =true;
    }

  default:
    break;
  }

  return handled;
}

