/*
 * Worldcup is Copyright (c) 2001-2006 by Thomas Pundt
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without a written agreement is hereby granted,
 * provided that the above copyright notice and this paragraph and the
 * following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 * THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <PalmOS.h>

#include "worldcup.h"
#include "worldcuprcp.h"

void* Id2Ptr(UInt16 id)
{ 
  FormPtr frm = FrmGetActiveForm();
  return FrmGetObjectPtr(frm, FrmGetObjectIndex(frm,id));
}

void ButtonSetLabel(Int16 id, char *l)
{
  ControlType *ctlP = Id2Ptr(id);
  char *label;
  if (l) {
    label = (char*)CtlGetLabel(ctlP);
    CtlEraseControl(ctlP);
    StrCopy(label,l);
    CtlDrawControl(ctlP);
  }
}

char* GetString(UInt16 id)
{
  char *p;
  MemHandle h = DmGetResource('tSTR', id);
  if (!h) return NULL;
  p = MemHandleLock(h);
  MemHandleUnlock(h);
  return p;
}

void DisplayErrorMessage(UInt16 id)
{
  char *msg = GetString(id);
  FrmCustomAlert(IdAlertError,msg,NULL,NULL);
}

void DisplayInformationMessage(UInt16 id)
{
  char *msg = GetString(id);
  FrmCustomAlert(IdAlertInformation,msg,NULL,NULL);
}

UInt16 DisplayConfirmationMessage(UInt16 id)
{
  char *msg = GetString(id);
  return FrmCustomAlert(IdAlertConfirmation,msg,NULL,NULL);
}

void SetHandleInt(FieldType* fptr, Int32 n)
{
  MemHandle h;
  if (!(h = FldGetTextHandle(fptr))) {
    h = MemHandleNew(7);
  }
  if (n==-1)
    StrCopy(MemHandleLock(h),"");
  else
    StrIToA(MemHandleLock(h), n);
  MemHandleUnlock(h);
  FldSetTextHandle(fptr, h);
}

void SetTextFieldWithHandle(FieldPtr textFldP, MemHandle textH, Boolean drawIt)
{
  MemHandle oldTextH =  FldGetTextHandle(textFldP);
  FldSetTextHandle(textFldP,textH);
  if (drawIt)
    FldDrawField(textFldP);
  if (oldTextH)
    MemHandleFree(oldTextH);
}

Err SetTextFieldWithString(FieldPtr textFldP, char *string, Boolean drawIt)
{
  Err anErr = errNone;
  MemHandle textH = MemHandleNew(StrLen(string)+sizeof(Char));
  if (textH) {
    StrCopy(MemHandleLock(textH),string);
    MemHandleUnlock(textH);
    SetTextFieldWithHandle(textFldP,textH,drawIt);
  } else {
    anErr = memErrNotEnoughSpace;
  }
  return anErr;
}

/*
 * Graphical buttons are only available since OS 3.5;
 * we emulate them using WinDrawBitmap().
 * Parameters: res        Resource ID of control;
 *             bitmapid   Id of bitmap
 */
void DrawButton(Int16 res, Int16 bitmapid)
{
  RectangleType rect;
  FormPtr frm;
  void *hBmp;

  frm = FrmGetActiveForm();
  FrmGetObjectBounds(frm,FrmGetObjectIndex(frm,res),&rect);
  CtlShowControl(Id2Ptr(res));
  if ( (hBmp = DmGetResource((UInt32)'Tbmp',bitmapid)) ) {
    WinDrawBitmap(MemHandleLock(hBmp),
                  rect.topLeft.x,rect.topLeft.y);
    MemHandleUnlock(hBmp);
    DmReleaseResource(hBmp);
  }
}

/*
 * Reposition/recalculate scrollbar for a field.
 */
void FldRecalculateScrollbar(FieldPtr fld, ScrollBarPtr scl)
{
  UInt16 scrollPos, textHeight, fieldHeight;
  Int16 maxScrollPos;
  
  // Get the current scroll position of the field, the total number of
  // lines in the field, and the number of lines visible in the field
  FldGetScrollValues(fld, &scrollPos, &textHeight, &fieldHeight);
  
  // Calculate the maximum scroll position:
  if(textHeight > fieldHeight)
    maxScrollPos = textHeight - fieldHeight + 1;
  else
    maxScrollPos = scrollPos;
  
  // Set the scroll bar's position:
  // Use the page size fieldHeight-1; this sets the size of the scroll car,
  // and gives an overlap of 1 line when page scrolling.
  SclSetScrollBar(scl, scrollPos, 0, maxScrollPos, fieldHeight-1);
}

/*
 * Do all the computing and scrolling.
 */
void FldDoScrolling(FieldPtr fld, ScrollBarPtr scl)
{
  Int16 value, min, max, pageSize;
  UInt16 scrollPos, textHeight, fieldHeight;
        
  // Get the current position of the scroll bar:
  SclGetScrollBar(scl, &value, &min, &max, &pageSize);
  
  // Get the current scroll position of the field:
  FldGetScrollValues(fld, &scrollPos, &textHeight, &fieldHeight);
  
  // Scroll the field according to the scroll bar's position:
  if(scrollPos > value)
    FldScrollField(fld, scrollPos-value, winUp);
  else if(scrollPos < value)
    FldScrollField(fld, value-scrollPos, winDown);
}

/*
 * Show a modal dialog; set event handler and use a init function if given.
 */
void MyDoDialog(UInt16 frmID, 
		FormEventHandlerType *evtHandler,
		FormInitFunctionType *initForm,
		UInt16 msgID,
		UInt16 titleID)
{
  FormPtr savedForm = FrmGetActiveForm(),
          frm = FrmInitForm(frmID);
  FrmSetActiveForm(frm);
  if (initForm) initForm(msgID,titleID);
  if (evtHandler) FrmSetEventHandler(frm, evtHandler);
  FrmDrawForm(frm);
  FrmDoDialog(frm);
  FrmDeleteForm(frm);
  FrmSetActiveForm(savedForm);
}
