/*
 * Worldcup is Copyright (c) 2001-2006 by Thomas Pundt
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without a written agreement is hereby granted,
 * provided that the above copyright notice and this paragraph and the
 * following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 * THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <PalmOS.h>

#include "worldcup.h"
#include "worldcuprcp.h"

#include "datedb.h"

static struct {
  UInt32 date;
  UInt8 group;
  UInt8 round;
} rowDate[64];
static TablePtr matchesTable;
static UInt32 now, lastDay;

/**
 * Matches Form
 */
static void RedrawTable(TablePtr tblP, ScrollBarPtr sclP, UInt16 first, Int16 rows)
{
  Int16 row, last = first+10;
  Int16 max = rows-11;

  if (max<0) max=0;

  if (last>rows-1) last=rows-1;
  for (row=0; row<TblGetNumberOfRows(tblP); row++)
    TblSetRowUsable(tblP, row, (row>=first && row<=last) ? true : false);

  SclSetScrollBar(sclP,first,0,max,10);
  TblMarkTableInvalid(tblP);
  TblRedrawTable(tblP);
}

void SetMatchesListTeamString(Int16 num, char *p, char *buf)
{
  struct team *team;
  Int16 platz, gruppe;
  if ( num > 0) {
    if ( num <= 32 ) {
      p += (num-1)*sizeof(struct team);
      team = (struct team*) p;
      StrCopy(buf,team->name);
    } else {
      platz = (num-64)/10;
      gruppe = (num-64)%10;
      p = GetString(IdDefaultName);
      StrIToA(buf,platz);
      StrCat(buf,". Gr ");
      // StrCat(buf,p);
      StrCat(buf,"  ");
      buf[StrLen(buf)-1] = 'A'-1+gruppe;
    }
  }
}

static void StripYearFromDate(char *buf)
{
  switch (PrefGetPreference(prefDateFormat)) {
  case dfMDYWithSlashes:
  case dfDMYWithSlashes:
  case dfDMYWithDashes:
    buf[StrLen(buf) - 3] = '\0';
    break;
  case dfDMYWithDots:
    buf[StrLen(buf) - 2] = '\0';
    break;
  case dfYMDWithSlashes:
  case dfYMDWithDots:
  case dfYMDWithDashes:
    MemMove(buf, buf + 3, StrLen(buf + 3) + 1);
    break;
  }
}

static void DrawMatchesItem(void* tableP, Int16 row, Int16 column, 
                            RectangleType *bounds)
{
  Int16 n, num;
  UInt16 x;
  Char buf[64], *p, *q;
  DateTimeType dateP;
  MemHandle handle;
  FontID fnt;
  UInt32 tmpDate;

  n = TblGetItemInt(tableP, row, column);
  x = bounds->topLeft.x;

  if (column<2 && rowDate[n].date >= now && rowDate[n].date<now+86400)
    fnt = FntSetFont(boldFont);
  else
    fnt = FntSetFont(stdFont);

  switch (column) {
  case 0:
    tmpDate = rowDate[n].date-(rowDate[n].date % 86400);
    if (lastDay != tmpDate) {
      TimSecondsToDateTime(n<64?(rowDate[n].date):0, &dateP);
      DateToAscii(dateP.month,dateP.day,dateP.year,
		  PrefGetPreference(prefDateFormat), buf);
      StripYearFromDate(buf);
      lastDay = tmpDate;
    } else {
      buf[0] = '\0';
    }
    break;

  case 1:
    TimSecondsToDateTime(n<64?(rowDate[n].date):0, &dateP);
    TimeToAscii(dateP.hour,dateP.minute,
		tfColon24h, buf);
    x += bounds->extent.x - FntCharsWidth(buf,StrLen(buf)) - 1;
    break;

  case 2:
    if ( (handle = DmQueryRecord(worldcupDB,3)) ) {
      p = q = MemHandleLock(handle);
      MemHandleUnlock(handle);

      if ( (num = (n%256)) < 112) {
	SetMatchesListTeamString(n>>8,p,buf);      
	StrCat(buf,"-");
	SetMatchesListTeamString(num,q,buf+StrLen(buf));
      } else {
        if (num<120) {
	  StrCopy(buf,GetString(IdRoundString+2));
	} else if (num<125) {
	  StrCopy(buf,GetString(IdRoundString+3));
	} else {
	  StrCopy(buf,GetString(IdRoundString+4));
	}
      }

    }
    break;

  case 3:
    if (n == -1) {
      StrCopy(buf, "");
    } else {
      StrIToA(buf, n);
    }
    StrCat(buf," :");
    x += bounds->extent.x - FntCharsWidth(buf,StrLen(buf)) - 1;
    break;

  case 4:
    if (n == -1) {
      StrCopy(buf, "");
    } else {
      StrIToA(buf, n);
    }
    break;
  }

  WinSetClip(bounds);
  WinDrawChars(buf, StrLen(buf), x, bounds->topLeft.y);
  WinResetClip();
  FntSetFont(fnt);
}  

/*
 * Arguments: 
 * secs: datetime of match
 * n:    encoded match
 */
void MakeDatebookEntry(UInt32 secs, Int16 n)
{
  ApptDBRecordType dbrec;
  ApptDateTimeType dbwhen;
  AlarmInfoType dbalarm;
  Int16 num;
  MemHandle handle;
  DateTimeType date_time;
  UInt16 index;

  MemSet(&dbrec, sizeof(ApptDBRecordType), 0);
  MemSet(&dbwhen, sizeof(ApptDateTimeType), 0);
  MemSet(&dbalarm, sizeof(AlarmInfoType), 0);

  dbrec.when = &dbwhen;
  dbrec.alarm = &dbalarm;
  dbrec.repeat = NULL;
  dbrec.note = NULL;
  dbrec.description = NULL;

  if ( (handle = DmQueryRecord(worldcupDB,3)) ) {
    Char buf[64], *p, *q;
    p = q = MemHandleLock(handle);
    MemHandleUnlock(handle);
    
    if ( (num = (n%256)) < 112) {
      SetMatchesListTeamString(n>>8,p,buf);      
      StrCat(buf,"-");
      SetMatchesListTeamString(num,q,buf+StrLen(buf));
    } else {
      if (num<120) {
	StrCopy(buf,GetString(IdRoundString+2));
      } else if (num<125) {
	StrCopy(buf,GetString(IdRoundString+3));
      } else {
	StrCopy(buf,GetString(IdRoundString+4));
      }
    }
    dbrec.description = buf;
  }
  
  DateSecondsToDate(secs,&dbwhen.date);
  TimSecondsToDateTime(secs, &date_time);
  if (secs % 60)
    TimAdjust(&date_time, (60 - (secs % 60)));
  dbwhen.startTime.hours = date_time.hour;
  dbwhen.startTime.minutes = date_time.minute;
  if (dbwhen.startTime.hours < 22) {
    dbwhen.endTime.hours = dbwhen.startTime.hours+2;
    dbwhen.endTime.minutes = dbwhen.startTime.minutes;
  } else {
    dbwhen.endTime.hours = 23;
    dbwhen.endTime.minutes = 59;
  }

  dbalarm.advanceUnit = aauMinutes;
  dbalarm.advance = 0;

  ApptNewRecord(datebook, &dbrec, &index);
}

static Int16 InitFormMatches(void)
{
  Int16 col, row, vis = 0;
  static Int16 widthDate = -1, widthTime;
  char *p, buf[10];
  struct match Match;
  DateTimeType dateP;
  FormPtr frm = FrmGetActiveForm();

  now = TimGetSeconds();
  now -= now % 86400;
  lastDay = 0;
  matchesTable = Id2Ptr(IdFrmMatches);

  if (widthDate == -1) {
    FontID fnt = FntSetFont(boldFont);
    TimSecondsToDateTime(3105721800UL, &dateP);
    DateToAscii(dateP.month,dateP.day,dateP.year,
                PrefGetPreference(prefDateFormat), buf);
    StripYearFromDate(buf);
    widthDate = FntCharsWidth(buf,StrLen(buf))+1;
    TimeToAscii(dateP.hour,dateP.minute,tfColon24h, buf);
    widthTime = FntCharsWidth(buf,StrLen(buf))+1;
    FntSetFont(fnt);
  }
  TblSetColumnWidth(matchesTable,0,widthDate);
  TblSetColumnWidth(matchesTable,1,widthTime);
  TblSetColumnWidth(matchesTable,2,129-(widthDate+widthTime));
  
  for (row=0; row<TblGetNumberOfRows(matchesTable); row++) {
    TblSetRowSelectable(matchesTable, row, true);
    TblSetRowUsable(matchesTable, row, false);
    for (col=0; col<=4; col++) {
      TblSetItemStyle(matchesTable, row, col, customTableItem);
      TblSetColumnUsable(matchesTable, col, true);
      TblSetCustomDrawProcedure(matchesTable, col, DrawMatchesItem);
    }
  }

  for ( p=ReadFixture(0,&Match),row=0; p; p=ReadFixture(p,&Match),row++ ) {
    if (row>63) continue;
    TblSetItemInt(matchesTable, row, 0, row);
    TblSetItemInt(matchesTable, row, 1, row);
    rowDate[row].date  = Match.date+appPrefs.gmtOff;
    rowDate[row].group = Match.group;
    rowDate[row].round  = 0;
    if (Match.team1>48) rowDate[row].round = 1;
    if (Match.team1>56) rowDate[row].round = 2;
    if (Match.team1>60) rowDate[row].round = 3;
    if (rowDate[row].date < now) vis=row+1;

    if (Match.group>0)
      TblSetItemInt(matchesTable, row, 2, 
		    ((Match.team1+(Match.group-1)*4)<<8)
		    + Match.team2+(Match.group-1)*4);
    else
      TblSetItemInt(matchesTable, row, 2, 
		    (ComputeSetHandleString(0, Match.team1)<<8)
		    + ComputeSetHandleString(0, Match.team2));
    TblSetItemInt(matchesTable, row, 3, Match.r1);
    TblSetItemInt(matchesTable, row, 4, Match.r2);
  }

  if (vis>54) vis=54;
  for (row=vis; row<vis+10; row++) {
    TblSetRowUsable(matchesTable, row, true);
  }

  FrmDrawForm(frm);

  DrawButton(IdFrmMatches+4,IdBitmap+10);
  DrawButton(IdFrmMatches+5,IdBitmap+13);
  DrawButton(IdFrmMatches+6,IdBitmap+14);

  return vis;
}

Boolean MatchesFormHandleEvent(EventPtr e)
{
  Boolean handled;
  static Int16 row;
  static ScrollBarPtr scrollP;

  handled = false;
  switch (e->eType) {

  case frmOpenEvent:
    row = InitFormMatches();
    scrollP = Id2Ptr(IdFrmMatches+1);
    RedrawTable(matchesTable, scrollP, row, 64);
    handled = true; 
    break;

  case menuEvent:

    switch(e->data.menu.itemID) {

    case IdMenuItem:
      FrmPopupForm(IdFrmAbout);
      break;

    case IdMenuItem+1:
      MyDoDialog(IdFrmHelp, FrmHelpHandleEvent, InitFrmHelp, IdFrmMatches, IdStringTitle+1);
      break;

    case IdMenuItem+2:
      FrmPopupForm(IdFrmTimezone);
      break;

    case IdMenuItem+3:
      FrmGotoForm(IdFrmMain);
      break;

    case IdMenuItem+5:
      BeamDatabase();
      break;

    case IdMenuItem+6:
      FrmGotoForm(IdFrmFixture);
      break;

    }
    handled = true;
    break;

  case tblSelectEvent:
    // tap on the date fields
    if ( e->data.tblSelect.column == 0 || e->data.tblSelect.column == 1 ) {
      if ( 0 == DisplayConfirmationMessage(IdErrorMsg+4)) {
        if (OpenDatebook()) {
	  MakeDatebookEntry(rowDate[e->data.tblSelect.row].date,
			    TblGetItemInt(matchesTable, e->data.tblSelect.row, 2));
	  DmCloseDatabase(datebook);
	}
      }
      TblUnhighlightSelection(matchesTable);
    } else {
      appPrefs.group = rowDate[e->data.tblSelect.row].group;
      appPrefs.round = rowDate[e->data.tblSelect.row].round;
      FrmGotoForm(IdFrmMain);
    }

    handled = true;
    break;

  // Scrollbar Event
  case sclRepeatEvent:
    row = e->data.sclRepeat.newValue;
    RedrawTable(matchesTable, scrollP, row, 64);
    break;

  case ctlSelectEvent:
    switch (e->data.ctlSelect.controlID) {
    case IdFrmMatches+4: // "Group form" button
      FrmGotoForm(IdFrmMain);
      break;
    case IdFrmMatches+6: // "Overview form" button
      FrmGotoForm(IdFrmFixture);
      break;
    }
    handled = true;
    break;

  case keyDownEvent:
    if (e->data.keyDown.chr == vchrPageUp) {
      row -= 11;
      if (row<0) row = 0;
      RedrawTable(matchesTable, scrollP, row, 64);
      handled = true;
    }

    if (e->data.keyDown.chr == vchrPageDown) {
      row += 11;
      if (row>53) row = 53;
      RedrawTable(matchesTable, scrollP, row, 64);
      handled = true;
    }

    if (e->data.keyDown.chr == vchrRockerLeft ||
	e->data.keyDown.keyCode == fiveWayLeft) {
      FrmGotoForm(IdFrmMain);
      handled = true;
    }

    if (e->data.keyDown.chr == vchrRockerRight || 
	e->data.keyDown.keyCode == fiveWayRight) {
      FrmGotoForm(IdFrmFixture);
      handled = true;
    }

    break;

  default:
    break;
  }

  return handled;
}

