/*
 * Worldcup is Copyright (c) 2001-2006 by Thomas Pundt
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without a written agreement is hereby granted,
 * provided that the above copyright notice and this paragraph and the
 * following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 * THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <PalmOS.h>

#include "worldcup.h"
#include "worldcuprcp.h"

DmOpenRef worldcupDB, worldcupPics, datebook;
struct prefs thePrefs;

static LocalID theId;
static UInt16  theCard;

Boolean OpenDatebook()
{
  LocalID id = 0;
  DmSearchStateType state; 
  UInt16 card;
  Err theErr;

  /*
   * Try to open the datebook database
   */
  datebook = 0;
  theErr = DmGetNextDatabaseByTypeCreator(true, &state, 'DATA', 
					  'date', true, &card, &id); 
  if ( (theErr != dmErrCantFind) ) {
    datebook = DmOpenDatabase(card, id, dmModeReadWrite);
    return true;
  }
  return false;
}

Boolean OpenDatabase()
{
  LocalID id = 0;
  DmSearchStateType state; 
  UInt16 card;
  Err theErr;
  MemHandle handle;
  char *p;

  /*
   * Open the main database; it includes the schedule, team names etc.
   */
  theErr = DmGetNextDatabaseByTypeCreator(true, &state, 'Data', 
					  'tpWC', true, &card, &id); 

  if ( (theErr == dmErrCantFind) || 
       (0 == (worldcupDB = DmOpenDatabase(card, id, dmModeReadWrite)) ) ) {
    DisplayErrorMessage(IdErrorMsg);
    return false;
  }

  if (! (handle = DmQueryRecord(worldcupDB, 0)) ) {
    DisplayErrorMessage(IdErrorMsg+1);
    DmCloseDatabase(worldcupDB);
    return false;
  }

  theId = id;
  theCard = card;

  p = MemHandleLock(handle);
  MemMove(&thePrefs, p, sizeof(thePrefs));
  MemHandleUnlock(handle);
  
  if (thePrefs.version != 4) {
    DisplayErrorMessage(IdErrorMsg+2);
    DmCloseDatabase(worldcupDB);
    return false;
  }

  /*
   * Try to open the optional Pictures database
   */
  theErr = DmGetNextDatabaseByTypeCreator(true, &state, 'Pics', 
					  'tpWC', true, &card, &id); 
  if ( (theErr != dmErrCantFind) )
    worldcupPics = DmOpenDatabase(card, id, dmModeReadOnly);
  
  return true;
}

char* ReadFixture(char* pos, struct match* match)
{
  MemHandle handle;
  static char *start;
  static UInt32 size;
  if (!pos) {
    handle = DmQueryRecord(worldcupDB,1);
    start = pos = MemHandleLock(handle);
    size = MemHandleSize(handle);
  } else {
    pos += sizeof(struct match);
  }
  if (pos-start>=size) {
    MemPtrUnlock(start);
    return 0;
  }
  MemMove(match, pos, sizeof(struct match));
  return pos;
}

void GetMatch(UInt8 num, struct match* match)
{
  char *p = MemHandleLock(DmQueryRecord(worldcupDB,1));
  MemMove(match, p+sizeof(struct match)*num, sizeof(struct match));
  MemPtrUnlock(p);
  return;
}

void WriteFixture(struct matchfield *match, UInt8 num, UInt8 group)
{
  struct match Match;
  int i;
  char *p = MemHandleLock(DmGetRecord(worldcupDB,1));

  for (i=0; i<num; i++) {
    Match.group = group;
    Match.date = match[i].date;
    Match.venue = match[i].venue;
    Match.r1 = match[i].r1;
    Match.r2 = match[i].r2;
    Match.team1 = match[i].team1;
    Match.team2 = match[i].team2;
    DmWrite(p, match[i].recNo*sizeof(struct match), 
	    &Match, sizeof(struct match));
  }
  MemPtrUnlock(p);
  DmReleaseRecord(worldcupDB,1,true);
}

void SetHandleString(FieldType *fptr, int num)
{
  MemHandle handle;
  char *p;
  struct team *team;

  if ( (handle = DmQueryRecord(worldcupDB,3)) ) {
    p = MemHandleLock(handle);
    MemHandleUnlock(handle);
    p += num*sizeof(struct team);
    team = (struct team*) p;
    SetTextFieldWithString(fptr, team->name, false);
  }

}

struct team* GetTeamString(int num)
{
  MemHandle handle;
  char *p;
  struct team *team;

  if ( num && (handle = DmQueryRecord(worldcupDB,3)) ) {
    p = MemHandleLock(handle);
    p += (num-1)*sizeof(struct team);
    team = (struct team*) p;
    MemHandleUnlock(handle);
    return team;
  }
  return 0;
}

UInt8 ComputeSetHandleString(FieldType *fptr, int num)
{
  char *p, buffer[20];
  struct matchfield match[6];
  struct teamstate table[4];
  struct match Match;
  int pos, group, final, i, j, rec, newnum;
  int onpos[4] = { -1, -1, -1, -1 };

  if (num>10 && num<30) {
    pos = num / 10;
    group = num % 10;
    ReadSetFixture(match,group,false);
    final = ComputeTable(table,match,15,0);
    if (!final) {
      if (fptr) {
	p = GetString(IdDefaultName);
	StrIToA(buffer,pos);
	StrCat(buffer,". ");
	StrCat(buffer,p);
	StrCat(buffer,"  ");
	buffer[StrLen(buffer)-1] = 'A'-1+group;
	SetTextFieldWithString(fptr,buffer,false);
      }
      // indicates that the finalalists are yet unknown
      return num+64;
    }

    // this might be stored easier...
    for (i=0; i<4; i++) {
      for (j=table[i].Pos-1; j<4; j++) { 
	if ( onpos[j] == -1 ) {
	  onpos[j] = i;
	  break;
	}
      }
    }

    if (fptr)
      SetHandleString(fptr, (group-1)*4+onpos[pos-1]);
    return (group-1)*4+onpos[pos-1]+1;
  } 

  if (num>48 && num<165) {
    pos = num / 100 + 1;
    rec   = num % 100 - 1;
    GetMatch(rec,&Match);
    newnum = num;
    if (pos == 1) {
      if (Match.r1 > Match.r2) {
	newnum = Match.team1;
      }
      if (Match.r1 < Match.r2) {
	newnum = Match.team2;
      }
    }
    if (pos == 2) {
      if (Match.r1 > Match.r2) {
	newnum = Match.team2;
      }
      if (Match.r1 < Match.r2) {
	newnum = Match.team1;
      }
    }
    if (newnum<num) 
      return ComputeSetHandleString(fptr,newnum);
    else {
      if (fptr) {
	p = GetString(IdDefaultName+1);
	StrIToA(buffer,pos);
	StrCat(buffer,". ");
	StrCat(buffer,p);
	StrCat(buffer," ");
	StrIToA(buffer+StrLen(buffer),rec+1);
	SetTextFieldWithString(fptr,buffer,false);
      }
    }
  }
  return rec+64;
}

struct venue* GetLocation(int num)
{
  MemHandle handle;
  char *p;
  struct venue *venue;

  if ( (handle = DmQueryRecord(worldcupDB,2)) ) {
    p = MemHandleLock(handle);
    p += num*sizeof(struct venue);
    venue = (struct venue*) p;
    MemHandleUnlock(handle);
    return venue;
  }
  return 0;
}

UInt16 readPlayers(char *items[], UInt16 selectedTeam)
{
  UInt16 i = 0;
  MemHandle handle;
  char *p, *end;

  if ( (handle = DmQueryRecord(worldcupDB,3+selectedTeam)) ) {
    p = MemHandleLock(handle);
    end = p+MemHandleSize(handle);
    while (p<end) {
      items[i++] = p;
      p += sizeof(struct player);
    }
    MemHandleUnlock(handle);
  }
  return i;
}

void SaveStatus(void)
{
  UInt16 dbAttr;
  if (errNone == DmDatabaseInfo(theCard,theId,NULL,&dbAttr,
				NULL,NULL,NULL,NULL,NULL,NULL,
				NULL,NULL,NULL)) {
    dbAttr |= dmHdrAttrBackup;
    DmSetDatabaseInfo (theCard,theId,NULL,&dbAttr,
		       NULL, NULL, NULL, NULL, NULL, NULL, 
		       NULL, NULL, NULL);
  }
  // close all previously opened databases
  if (worldcupDB)
    DmCloseDatabase(worldcupDB);
  if (worldcupPics)
    DmCloseDatabase(worldcupPics);
  PrefSetAppPreferences('tpWC', 0, appPrefs.version, 
			&appPrefs, sizeof(appPrefs), false);
}

/*
 * Beaming support
 */
static Err WriteDBData(const void *dataP, UInt32 *sizeP, void *userDataP)
{
  Err err;
  *sizeP = ExgSend((ExgSocketType*)userDataP,(void*)dataP,*sizeP,&err);
  return err;
}

void BeamDatabase(void)
{
  ExgSocketType s;
  char name[] = "worldcupDB.pdb";
  Err err;

  if ( !FeatureSet30 ) {
    DisplayInformationMessage(IdErrorMsg+5);
    return;
  }

  MemSet(&s, sizeof(s), 0);
  s.description = name;
  s.name = name;

  if (! (err = ExgPut(&s)) ) {
    err = ExgDBWrite(WriteDBData,&s,NULL,theId,theCard);
  }
  ExgDisconnect(&s, err);
}
